#! /usr/bin/env python
# encoding: utf-8

import os
import sys
from waflib import Options, Utils
from waflib.Configure import conf
from waflib.TaskGen import feature
###############################################################################
###############################################################################
# Support functions to define toolchains and devices
###############################################################################
###############################################################################

class toolchain(object):
    ''' a special decorator class used to wrap common actions around
        a toolchain configure func and toregister all defined toolchains'''
    toolchains = {}

    def __init__(self, name, *aliases):
        super(toolchain,self).__init__()
        self.name = name

        self.toolchains[self.name] = self

        for x in aliases:
            self.toolchains[x] = self

    def __call__(self,*args,**kwargs):
        configure_func =  getattr(self,'configure_function',None)

        if configure_func:
            return self.configure_toolchain(*args,**kwargs)
        else:
            # this will be invoked on the first call
            return self.setup_configure_func(*args,**kwargs)

    def setup_configure_func(self,func):
        self.configure_function = func
        return self

    def configure_toolchain(self,ctx):
        envname = 'toolchain_%s' % self.name

        if envname in ctx.all_envs:
            ctx.setenv(envname)
        else:
            ctx.setenv('')
            ctx.setenv(envname,ctx.env)
            ctx.env['TOOLCHAIN'] = self.name

            try:
                self.configure_function(ctx)
            except:
                ctx.setenv('')
                del ctx.all_envs[envname]
                raise

    def select_toolchain(self,ctx):
        envname = 'toolchain_%s' % self.name

        if envname in ctx.all_envs:
            ctx.setenv(envname)
        else:
            ctx.fatal('Toolchain %s not configured' % self.name)

@conf
def configure_toolchain(conf,name):
    global toolchain
    try:
        func = toolchain.toolchains[name]
    except KeyError:
        raise conf.errors.ConfigurationError('Unkown toolchain %s' % name)

    func(conf)

@conf
def select_toolchain(ctx,name):
    global toolchain
    try:
        func = toolchain.toolchains[name].select_toolchain
    except KeyError:
        raise ctx.errors.ConfigurationError('Unkown toolchain %s' % name)
    func(ctx)

class device(object):
    ''' a special decorator class used to wrap common actions around
        a device configure func and to register all know device setups'''
    devices = {}

    def __init__(self,devices,toolchains):
        super(device,self).__init__()

        devices = Utils.to_list(devices)
        self.name       = '_'.join(devices)
        self.toolchains = Utils.to_list(toolchains)

        global toolchain

        for x in self.toolchains:
            if x not in toolchain.toolchains:
                raise Exception('Unknown Toolchain %s' % x)

        for x in devices:
            self.devices[x] = self

    def __call__(self,*args,**kwargs):
        configure_func =  getattr(self,'configure_function',None)

        if configure_func:
            return self.configure_device(*args,**kwargs)
        else:
            # this will be invoked on the first call
            return self.setup_configure_func(*args,**kwargs)

    def setup_configure_func(self,func):
        self.configure_function = func
        return self

    def configure_device(self,ctx,device, toolchain = None):
        toolchain = Utils.to_list(toolchain)

        for x in (toolchain or self.toolchains):
            try:
                ctx.configure_toolchain(x)
                break
            except ctx.errors.ConfigurationError:
                pass
        else:
            raise ctx.errors.ConfigurationError('Unable to configure a suitable toolchain for %s' % device)

        envname = 'device_%s_%s' % (self.name,ctx.env['TOOLCHAIN'])

        if envname in ctx.all_envs:
            ctx.setenv(envname)
        else:
            ctx.setenv(envname,ctx.env)
            ctx.env['DEVICE'] = self.name

            try:
                self.configure_function(ctx)
            except:
                ctx.setenv('')
                del ctx.all_envs[envname]
                raise

@conf
def configure_device(conf,name,toolchain = None):
    global device
    try:
        func = device.devices[name]
    except KeyError:
        raise conf.errors.ConfigurationError('Unkown device %s' % name)
    func(conf,name, toolchain)

@conf
def apply_device_flags(ctx,name):
    '''Function to update the current environment with device
       specific flags. This function is meant for backwards compat'''
    global device

    try:
        func = device.devices[name]
    except KeyError:
        raise ctx.errors.ConfigurationError('Unkown device %s' % name)

    func.configure_function(ctx)
    ctx.env.append_value('DEFINES', '__' + name.upper())

###############################################################################
###############################################################################
# Definition of standard toolchains and devices
###############################################################################
###############################################################################


@toolchain('hitex')
def configure_toolchain_hitex(conf):
    path = ""
    if 'PATH_GNU_ARM' in os.environ:
        path = os.path.join(os.environ['PATH_GNU_ARM'], 'bin')
        conf.env.env = dict(os.environ)
        conf.env.env['PATH'] = os.environ['PATH'] + os.pathsep + path

    conf.env.stash()
    try:
        conf.env['TOOL_PREFIX'] = 'arm-hitex-elf-'
        conf.setup_compiler(path_list = [path])
    except:
        conf.env.revert()
        raise

    conf.gcc_arm_flags()

@toolchain('codesourcery')
def configure_toolchain_codesourcery(conf):
    # get path to toolchain
    path = None

    if 'CS_PATH' in os.environ:
      path = [os.path.join(os.environ['CS_PATH'],'bin')]
    else:
      if sys.platform.startswith("win"):
        try:
          import _winreg as winreg
        except:
          import winreg

        aReg = winreg.ConnectRegistry(None, winreg.HKEY_LOCAL_MACHINE)
        aKey = None

        try:
          aKey = winreg.OpenKey(aReg, r"SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\Sourcery G++ Lite for ARM EABI")
        except:
          # Try Wow6432-Node, as Python x64 looks in native 64bit keys
          aKey = winreg.OpenKey(aReg, r"SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\Sourcery G++ Lite for ARM EABI", 0, winreg.KEY_READ | winreg.KEY_WOW64_32KEY)
        finally:
          if aKey is not None:
              try:
                  install_loc, type = winreg.QueryValueEx(aKey, "InstallLocation")
                  assert type is winreg.REG_SZ
                  if sys.version_info[0] == 2:
                    install_loc = install_loc.encode('ascii', 'ignore')
                  path = [os.path.join(install_loc, 'bin')]
              except EnvironmentError:
                  pass

              winreg.CloseKey(aKey)

        winreg.CloseKey(aReg)


    conf.env.stash()
    try:
        conf.env['TOOL_PREFIX'] = 'arm-none-eabi-'
        conf.setup_compiler(path_list=path)
    except:
        conf.env.revert()
        raise

    conf.gcc_arm_flags()

    f = conf.env.append_value

    f('CFLAGS',    ['-ffunction-sections', '-fdata-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('CXXFLAGS',  ['-ffunction-sections', '-fdata-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('ASFLAGS',   ['-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('LINKFLAGS', ['-Wl,-gc-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])


@toolchain('custom')
def configure_toolchain_custom(conf):
    opt = Options.options

    conf.env.stash()
    try:
        conf.env['TOOL_PREFIX'] = opt.custom_toolchain_prefix + '-'
        conf.setup_compiler(path_list=[opt.custom_toolchain_path],
                            compiler=opt.custom_toolchain_driver,
                            cxxcompiler=opt.custom_toolchain_cxxdriver)
    except:
        conf.env.revert()
        raise

    conf.gcc_arm_flags()

    f = conf.env.append_value

    f('CFLAGS',    ['-ffunction-sections', '-fdata-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('CXXFLAGS',  ['-ffunction-sections', '-fdata-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('ASFLAGS',   ['-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('LINKFLAGS', ['-Wl,-gc-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])

@toolchain('arm-eabi')
def configure_toolchain_arm_eabi(conf):
    exc = None
    for x in 'arm-none-eabi- arm-eabi-'.split():
        conf.env.stash()
        try:
            conf.env['TOOL_PREFIX'] = x
            conf.setup_compiler()
            break
        except conf.errors.ConfigurationError as e:
            exc = e
            conf.env.revert()
    else:
        raise exc

    conf.gcc_arm_flags()

    f = conf.env.append_value

    f('CFLAGS',    ['-ffunction-sections', '-fdata-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('CXXFLAGS',  ['-ffunction-sections', '-fdata-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('ASFLAGS',   ['-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])
    f('LINKFLAGS', ['-Wl,-gc-sections', '-msoft-float', '-mfpu=vfp', '-mfloat-abi=soft',])


@toolchain('llvm_xpic')
def configure_toolchain_xpic(conf):
    path = None
    if 'PATH_GNU_XPIC' in os.environ:
        path = [os.path.join(os.environ['PATH_GNU_XPIC'], 'bin')]

    conf.env.stash()
    try:
        conf.env['TOOL_PREFIX'] = 'xpic-'
        conf.env['MACHINE']     = 'xpic'
        conf.setup_compiler(compiler='llvmc',path_list=path)
    except:
        conf.env.revert()
        raise

    conf.env['cprogram_PATTERN'] = '%s.elf'

    f = conf.env.append_value

    basepath = os.path.abspath(os.path.dirname(conf.env["LINK_CC"][0]) + os.path.sep + ".." + os.path.sep)
    f("STLIBPATH", os.path.join(basepath,"lib"))
    f("STLIBPATH", os.path.join(basepath,"lib","gcc","xpic-hilscher-elf", "0.1.1"))


### ATTENTION: The order of the following entries matters!
###            the 'native' toolchain is overriden by the last
###            valid if branch!

if sys.platform in ("linux","linux2"):
    @toolchain('gcc', 'native')
    def configure_toolchain_gcc(conf):
        conf.env.stash()
        try:
            conf.env['TOOL_PREFIX'] = ''
            conf.setup_compiler()
        except:
            conf.env.revert()
            raise

        conf.env.append_value('CFLAGS',    [])
        conf.env.append_value('CXXFLAGS',  [])
        conf.env.append_value('ASFLAGS',   [])
        conf.env.append_value('LINKFLAGS', ['-Wl,-gc-sections'])

        conf.env['cprogram_PATTERN'] = '%s'

if sys.platform in ("win32","win64"):
    @toolchain('mingw32', 'mingw', 'native')
    def configure_toolchain_mingw32(conf):
        conf.env.stash()
        try:
            conf.env['TOOL_PREFIX'] = 'mingw32-'
            conf.setup_compiler(compiler='gcc')
        except:
            conf.env.revert()
            raise

        conf.env.append_value('CFLAGS',    [])
        conf.env.append_value('CXXFLAGS',  [])
        conf.env.append_value('ASFLAGS',   [])
        conf.env.append_value('LINKFLAGS', ['-Wl,-gc-sections'])

        conf.env['cprogram_PATTERN'] = '%s.exe'

if sys.platform in ("win64"):
    @toolchain('mingw64', 'native')
    def configure_toolchain_mingw64(conf):
        conf.env.stash()
        try:
            conf.env['TOOL_PREFIX'] = 'mingw64-'
            conf.setup_compiler(compiler='gcc')
        except:
            conf.env.revert()
            raise

        conf.env.append_value('CFLAGS',    [])
        conf.env.append_value('CXXFLAGS',  [])
        conf.env.append_value('ASFLAGS',   [])
        conf.env.append_value('LINKFLAGS', ['-Wl,-gc-sections'])

        conf.env['cprogram_PATTERN'] = '%s.exe'

@conf
def cc_version(conf):
     return tuple(int(x) for x in conf.env['CC_VERSION'])

@conf
def gcc_flags(conf):
    f = conf.env.append_value

    for x in 'CFLAGS CXXFLAGS'.split():
        f(x,[ '-Wall',
              '-Wredundant-decls',
              '-Wno-inline',
              '-Winit-self'])

    f('ASFLAGS',[ '-Wall',
                  '-Wredundant-decls',
                  '-Wno-inline',
                  '-c'])

    f('DEFINES_compile_release',  ['NDEBUG'])

    for x in 'CFLAGS CXXFLAGS'.split():
        f(x + '_check_ansi', ['-ansi',
                              '-pedantic'])

    for x in 'CFLAGS CXXFLAGS'.split():
        f(x + '_check_extra', ['-Wconversion',
                               '-Wmissing-field-initializers',
                               '-Wsign-compare',
                               '-Wpointer-arith',
                               '-Wcast-qual'])

    if conf.cc_version() >= (4,1,2):
        f('CFLAGS_check_extra', ['-Wc++-compat'])

    # define CFLAGS_warninglevel1 and CXXFLAGS_warninglevel2
    for x in 'CFLAGS CXXFLAGS'.split():
        f(x + '_warninglevel1', [ "-Wsystem-headers",
                                  "-Wbad-function-cast",
                                  "-Wsign-compare",
                                  "-Waggregate-return",
                                  "-Wswitch-default",
                                  "-Wstrict-prototypes",
                                  "-Wunreachable-code",
                                  '-Wpointer-arith',
                                  ])


@feature("check_extra", "check_ansi", "warninglevel1")
def compile_check_dummy(self):
    # make waf happy
    pass

@conf
def gcc_arm_flags(conf):
    f = conf.env.append_value

    f('CFLAGS_compile_arm',   ['-marm'])
    f('CXXFLAGS_compile_arm',   ['-marm'])
    f('LINKFLAGS_compile_arm',   ['-marm'])
    f('CFLAGS_compile_thumb', ['-mthumb'])
    f('CXXFLAGS_compile_thumb', ['-mthumb'])
    f('LINKFLAGS_compile_thumb', ['-mthumb'])

    conf.env['cprogram_PATTERN'] = '%s.elf'


@conf
def gcc_netx_flags(conf):
    conf.gcc_flags()

    f = conf.env.append_value

    for x in 'CFLAGS CXXFLAGS'.split():
        f(x + '_compile_debug',    ['-O0', '-g', '-gdwarf-2'])
        f(x + '_compile_debugrel', ['-Os', '-g', '-gdwarf-2'])
        f(x + '_compile_release',  ['-Os'])

    f('ASFLAGS_compile_debug',    ['-Wa,-gdwarf2'])
    f('ASFLAGS_compile_debugrel', ['-Wa,-gdwarf2'])

    for x in 'CFLAGS CXXFLAGS'.split():
        f(x,[ '-mlong-calls',
              '-mapcs',
              '-mthumb-interwork',
              '-fshort-enums',
              '-fno-common'])

    f('ASFLAGS',[ '-mapcs',
                  '-mthumb-interwork',
                  '-fshort-enums',
                  '-c'])

    f('LINKFLAGS', [ '-mthumb-interwork', '-nostdlib'])

    f('DEFINES', ['_NETX_'])
    f('STLIB_nxo_standardlibs', ['m', 'gcc'])
    f('STLIB_default_standardlibs',   ['m', 'c', 'gcc'])


@device('netx', 'hitex codesourcery arm-eabi')
def configure_device_netx(conf):
    conf.gcc_netx_flags()

    for x in 'CFLAGS CXXFLAGS ASFLAGS LINKFLAGS'.split():
        conf.env.append_value(x, '-march=armv5te')


@device('netx50','hitex codesourcery arm-eabi')
def configure_device_netx50(conf):
    conf.gcc_netx_flags()

    if conf.cc_version()[0:2] < (4,1):
        mcpu = '-mcpu=arm9e'
    else:
        mcpu = '-mcpu=arm966e-s'

    for x in 'CFLAGS CXXFLAGS ASFLAGS'.split():
        conf.env.append_value(x, mcpu)

    if conf.env.TOOLCHAIN == 'hitex':
        # Workaround for hitex linking
        # hitex compiler chooses hardware fp library whenever something
        # else than arm926ej-s is specified
        conf.env.append_value('LINKFLAGS', '-mcpu=arm926ej-s')
    else:
        conf.env.append_value('LINKFLAGS', mcpu)

@device('netx10 netx51 netx52','codesourcery arm-eabi')
def configure_device_netx51(conf):
    conf.gcc_netx_flags()

    if conf.cc_version()[0:2] < (4,1):
        mcpu = '-mcpu=arm9e'
    else:
        mcpu = '-mcpu=arm966e-s'

    for x in 'CFLAGS CXXFLAGS ASFLAGS'.split():
        conf.env.append_value(x, mcpu)

    if conf.env.TOOLCHAIN == 'hitex':
        # Workaround for hitex linking
        # hitex compiler chooses hardware fp library whenever something
        # else than arm926ej-s is specified
        conf.env.append_value('LINKFLAGS', '-mcpu=arm926ej-s')
    else:
        conf.env.append_value('LINKFLAGS', mcpu)

@device('netx100 netx500','hitex codesourcery arm-eabi')
def configure_device_netx100(conf):
    conf.gcc_netx_flags()

    for x in 'CFLAGS CXXFLAGS ASFLAGS LINKFLAGS'.split():
        conf.env.append_value(x, '-mcpu=arm926ej-s')

@device('xpic','llvm_xpic')
def configure_device_xpic(conf):
    f = conf.env.append_value

    for x in 'CFLAGS CXXFLAGS'.split():
        f(x + '_compile_debug',    ['-O0', '-g', '-gdwarf-s'])
        f(x + '_compile_debugrel', ['-Os', '-g', '-gdwarf-s'])
        f(x + '_compile_release', ['-Os'])

    f('ASFLAGS', ['-Wa,-mmcu=xpic','-c'])
    f('DEFINES', ['__XPIC__'])
    f('STLIB_default_standardlibs',   ['m', 'c', 'gcc'])

@device('xpic2','llvm_xpic')
def configure_device_xpic2(conf):
    f = conf.env.append_value

    for x in 'CFLAGS CXXFLAGS'.split():
        f(x + '_compile_debug',    ['-O0', '-g', '-gdwarf-s'])
        f(x + '_compile_debugrel', ['-Os', '-g', '-gdwarf-s'])
        f(x + '_compile_release', ['-Os'])

    f('ASFLAGS', ['-Wa,-mmcu=xpic2','-c'])
    f('DEFINES', ['__XPIC__'])
    f('STLIB_default_standardlibs',   ['m', 'c', 'gcc'])

if sys.platform in ("linux","linux2"):
    @device(('linux','native'),'gcc')
    def configure_device_linux(conf):
        conf.gcc_flags()

        f = conf.env.append_value

        for x in 'CFLAGS CXXFLAGS'.split():
            f(x + '_compile_debug',    ['-O0', '-g'])
            f(x + '_compile_debugrel', ['-O3', '-g'])
            f(x + '_compile_release', ['-O3'])

        f('STLIB',   ['m', 'c', 'gcc'])
        f('TOOL_OPTIONS', ["linkerscript_optional"])

if sys.platform in ("win32","win64"):
    @device(('win32','native'), 'mingw32')
    def configure_device_win32(conf):
        conf.gcc_flags()

        f = conf.env.append_value

        for x in 'CFLAGS CXXFLAGS'.split():
            f(x + '_compile_debug',    ['-O0', '-g',])
            f(x + '_compile_debugrel', ['-O3', '-g',])
            f(x + '_compile_release',  ['-O3'])

        f('ASFLAGS_compile_debug',    ['-Wa,-g'])
        f('ASFLAGS_compile_debugrel', ['-Wa,-g'])
        f('TOOL_OPTIONS', ["linkerscript_optional"])

if sys.platform == "win64":
    @device(('win64','native'), 'mingw64')
    def configure_device_win64(conf):
        conf.gcc_flags()

        f = conf.env.append_value

        for x in 'CFLAGS CXXFLAGS'.split():
            f(x + '_compile_debug',    ['-O0', '-g',])
            f(x + '_compile_debugrel', ['-O3', '-g',])
            f(x + '_compile_release',  ['-O3'])

        f('ASFLAGS_compile_debug',    ['-Wa,-g'])
        f('ASFLAGS_compile_debugrel', ['-Wa,-g'])
        f('TOOL_OPTIONS', ["linkerscript_optional"])
